/**
 * server-thread.c
 *
 * Програма моделює роботу багатопотокового паралельного сервера,
 * який здійснює обмін даними з клієнтами через потоковий сокет
 * у комунікаційному домені AF_INET. Сервер прив'язується до всіх
 * IP-адрес свого вузла і до порта, заданого константою SERVER_PORTNUM.
 * Робочі потоки створюються за необхідністю, по встановленню з'єднання
 * з черговим клієнтом. Кожен робочий потік обслуговує одного клієнта;
 * при цьому він у циклі отримує клієнтські повідомлення і 
 * без змін повертає їх назад клієнту (працює як сервер служби "Луна").
 *
 */

#include "myecho.h"

/*
 * Макрос _REENTRANT (див. <features.h>) указує на необхідність
 * використовувати реентерабельні (безпечні в багатопотоковому контексті)
 * варіанти реалізації деяких системних об'єктів, наприклад,
 * змінної errno.
 * Є специфічним для GNU/Linux (в стандарті SUS3 відсутній).
 * Потрібен для забезпечення максимальної сумісності 
 * середовища GNU/Linux з вимогами стандарту SUS3
 * стосовно підтримки багатопотоковості
 */
#define _REENTRANT

#include <assert.h>
#include <errno.h>
#include <netdb.h>
#include <pthread.h>
#include <semaphore.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <unistd.h>

/*
 * Журнал помилок
 */
#define ERRLOGFILE "server.err"

/*
 * Семафор для синхронізації головного та останнього з
 * робочих потоків (головний потік повинен дочекатись,
 * поки знов створений робочий потік прочитає значення
 * дескриптора з'єднаного з новим клієнтом сокета)
 */
sem_t conn_sem;

/*
 * Головна функція робочих потоків.
 * Обслуговує з'єднання з клієнтом
 */
void *handle_connection(void *arg);

int main()
{
	pthread_t thread_id;
	pthread_attr_t thread_attr;
	int s, conn_s;
	struct sockaddr_in serv_addr;
	int reuseaddr = 1;
	int efl;

	// Приєднує стандартний потік виведення повідомлень про помилки
	// до файлу ERRLOGFILE
	if ((freopen(ERRLOGFILE, "w", stderr)) == NULL)
		exit(EXIT_FAILURE);

	// Виконує ініціалізацію семафора conn_sem
	// (присвоюючи йому початкове значення 0)
	if (sem_init(&conn_sem, 0, 0) == -1)
		perror("sem_init()");

	// Виконує ініціалізацію контейнера потокових атрибутів
	efl = pthread_attr_init(&thread_attr);
	assert(efl == 0);
	// Атрибуту detachstate присвоює значення PTHREAD_CREATE_DETACHED
	// (створити потік як від'єднаний)
	efl = pthread_attr_setdetachstate(&thread_attr,
	                                  PTHREAD_CREATE_DETACHED);
	assert(efl == 0);

	// Створює потоковий сокет в домені AF_INET
	s = socket(AF_INET, SOCK_STREAM, 0);
	if (s < 0) {
		perror("socket()");
		exit(EXIT_FAILURE);
	}

	// Встановлює для сокета s опцію SO_REUSEADDR
	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR,
	    (void *) &reuseaddr, (socklen_t) sizeof(int)) != 0) {
		perror("setsockopt()");
		exit(EXIT_FAILURE);
	}

	// Розміщує в структурі serv_addr свою адресу
	memset(&serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_addr.s_addr = INADDR_ANY;	// Всі IP-адреси даного вузла
	// htons() переставляє байти свого аргумента згідно з мережевим порядком
	serv_addr.sin_port = htons(SERVER_PORTNUM);
	// Прив'язує сокет s до локальної адреси
	if (bind(s, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) != 0) {
		perror("bind()");
		exit(EXIT_FAILURE);
	}

	// Переводить сокет s в стан готовності до прийому запитів
	// на встановлення з'єднання (і створює для нього чергу запитів
	// на встановлення з'єднання довжиною 5)
	if (listen(s, 5) != 0) {
		perror("listen()");
		exit(EXIT_FAILURE);
	}

	while (1) {

		// Встановлює з'єднання з черговим клієнтом
		while (1) {
			conn_s = accept(s, NULL, NULL);
			if (conn_s >= 0)
				break;
			switch (errno) {
			case EINTR:
			case ECONNABORTED:
				continue;
			default:
				perror("accept()");
				exit(EXIT_FAILURE);
			}
		}

		// Створює робочий потік для обслуговування нового клієнта.
		// Робочий потік виконує функцію handle_connection(),
		// покажчик на conn_s передається їй як аргумент.
		// Згідно з досить широко розповсюдженою практикою в даній 
		// ситуації можна було б передавати не адресу, а значення conn_s
		// (приведене до типу void *). Такий прийом дозволив би обійтись 
		// у даній програмі без семафора. Однак, стандарт мови C 
		// не вимагає підтримки перетворення цілих величин
		// у покажчики і назад без втрати інформації. Хоча на практиці
		// таке перетворення завжди або майже завжди виконується
		// коректно (оскільки завжди або майже завжди виконується умова
		// sizeof(int) <= sizeof(void *))
		efl = pthread_create(&thread_id, &thread_attr,
		                     &handle_connection, &conn_s);
		if (efl != 0) {
			fprintf(stderr, "Error creating working thread: %s\n",
			        strerror(efl));
			if (close(conn_s) != 0)
				perror("close(conn_s)");
		} else
			// Чекає, доки новий потік прочитає значення змінної conn_s
			if (sem_wait(&conn_sem) != 0)
				perror("sem_wait");

	}
}

/*
 * Головна функція робочих потоків.
 * Обслуговує з'єднання з клієнтом.
 * arg повинен указувати на дескриптор зв'язаного
 * з клієнтом сокета.
 * Повертає завжди NULL 
 */
void* handle_connection(void* arg)
{
	int conn_fd;
	char buf[BUFSIZE];
	ssize_t n_recved;

	// Зберігає в змінній conn_fd дескриптор зв'язаного з клієнтом сокета
	conn_fd = *((int *) arg);
	// Розблоковує головний потік
	if (sem_post(&conn_sem) != 0)
		perror("sem_post()");
    
	// В  циклі отримує й повертає назад клієнтські повідомлення
	while (1) {
		n_recved = recv(conn_fd, buf, BUFSIZE, 0);
		if (n_recved > 0) {
			if (send(conn_fd, buf, n_recved, 0) < 0)
				perror("send()");
		} else
			break;
	}
	// Або помилка прийому, або клієнт закрив з'єднання
	if (n_recved < 0)
		perror("recv()");
	if (close(conn_fd) < 0)
		perror("close(conn_fd)");

	return NULL;
}
